<?php
require_once 'db_connection.php';

// Pagination settings
$limit = 15; // Records per page
$page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page > 0) ? ($page - 1) * $limit : 0;

// Get total number of approved transactions for pagination
$total_result = $conn->query("SELECT COUNT(*) as count FROM transactions WHERE status = 'approved'");
$total_rows = $total_result->fetch_assoc()['count'];
$total_pages = ceil($total_rows / $limit);

// Fetch approved transactions for the current page
$sql = "SELECT t.amount, t.payment_method, t.address, t.created_at, u.name as user_name, u.profile_picture as user_avatar 
        FROM transactions t 
        JOIN users u ON t.user_id = u.id 
        WHERE t.status = 'approved' 
        ORDER BY t.created_at DESC 
        LIMIT ? OFFSET ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ii", $limit, $offset);
$stmt->execute();
$result = $stmt->get_result();
$transactions = [];
while($row = $result->fetch_assoc()) {
    $transactions[] = $row;
}
$stmt->close();
$conn->close();

function getAvatarUrl($user) {
    if (!empty($user['user_avatar'])) {
        return htmlspecialchars($user['user_avatar']);
    }
    $initial = strtoupper(substr($user['user_name'], 0, 1));
    return 'https://placehold.co/40x40/6366f1/ffffff?text=' . urlencode($initial);
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Proofs</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style> body { font-family: 'Inter', sans-serif; } </style>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto p-4 md:p-8 max-w-5xl">
        <div class="bg-white rounded-2xl shadow-xl p-8">
            <div class="flex justify-between items-center mb-6 border-b pb-4">
                <h1 class="text-3xl font-bold text-gray-800">Payment Proofs</h1>
                <a href="index.php" class="text-indigo-600 hover:text-indigo-800 font-semibold">← Back to Dashboard</a>
            </div>

            <div class="overflow-x-auto rounded-lg border">
                 <table class="min-w-full divide-y divide-gray-200">
                     <thead class="bg-gray-50">
                         <tr>
                             <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
                             <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                             <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Method</th>
                             <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Address</th>
                             <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                         </tr>
                     </thead>
                     <tbody class="bg-white divide-y divide-gray-200">
                         <?php if (empty($transactions)): ?>
                             <tr><td colspan="5" class="px-6 py-10 text-center text-gray-500">No approved payments found.</td></tr>
                         <?php else: ?>
                             <?php foreach ($transactions as $tx): ?>
                                 <tr>
                                     <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <div class="flex-shrink-0 h-10 w-10">
                                                <img class="h-10 w-10 rounded-full object-cover" src="<?php echo getAvatarUrl($tx); ?>" alt="User Avatar">
                                            </div>
                                            <div class="ml-4">
                                                <div class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($tx['user_name']); ?></div>
                                            </div>
                                        </div>
                                     </td>
                                     <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900"><?php echo number_format($tx['amount']); ?> Tokens</td>
                                     <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo htmlspecialchars($tx['payment_method']); ?></td>
                                     <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 truncate max-w-xs" title="<?php echo htmlspecialchars($tx['address']); ?>"><?php echo htmlspecialchars($tx['address']); ?></td>
                                     <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo date("d M, Y", strtotime($tx['created_at'])); ?></td>
                                 </tr>
                             <?php endforeach; ?>
                         <?php endif; ?>
                     </tbody>
                 </table>
            </div>

            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
            <div class="mt-6 flex items-center justify-between">
                <a href="?page=<?php echo max(1, $page - 1); ?>" 
                   class="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 <?php echo $page <= 1 ? 'pointer-events-none opacity-50' : ''; ?>">
                    Previous
                </a>
                <div class="hidden sm:block">
                    <p class="text-sm text-gray-700">
                        Page
                        <span class="font-medium"><?php echo $page; ?></span>
                        of
                        <span class="font-medium"><?php echo $total_pages; ?></span>
                    </p>
                </div>
                <a href="?page=<?php echo min($total_pages, $page + 1); ?>" 
                   class="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 <?php echo $page >= $total_pages ? 'pointer-events-none opacity-50' : ''; ?>">
                    Next
                </a>
            </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>

