<?php
require_once 'db_connection.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}
$user_id = $_SESSION['user_id'];

// সাইট-ব্যাপী সেটিংস লোড করুন
$settings = [];
$result = $conn->query("SELECT setting_key, setting_value FROM settings");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
}
$min_withdrawal_amount = (int)($settings['min_withdrawal_amount'] ?? 1000);


// Get current user's token balance
$stmt_user = $conn->prepare("SELECT tokens FROM users WHERE id = ?");
$stmt_user->bind_param("i", $user_id);
$stmt_user->execute();
$user_result = $stmt_user->get_result();
$user = $user_result->fetch_assoc();
$current_tokens = $user['tokens'] ?? 0;
$stmt_user->close();

// Get available payment methods
$payment_methods = [];
$result_methods = $conn->query("SELECT name FROM payment_methods ORDER BY name ASC");
while($row = $result_methods->fetch_assoc()) {
    $payment_methods[] = $row['name'];
}


$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $amount = (int)($_POST['amount'] ?? 0);
    $address = trim($_POST['address'] ?? '');
    $method = trim($_POST['payment_method'] ?? '');

    if ($amount <= 0) {
        $error = "Please enter a valid amount.";
    } elseif ($amount < $min_withdrawal_amount) {
        $error = "Minimum withdrawal amount is " . number_format($min_withdrawal_amount) . " tokens.";
    } elseif ($amount > $current_tokens) {
        $error = "You don't have enough tokens for this request.";
    } elseif (empty($address)) {
        $error = "Payment address cannot be empty.";
    } elseif (empty($method) || !in_array($method, $payment_methods)) {
        $error = "Please select a valid payment method.";
    } else {
        // Start a transaction
        $conn->begin_transaction();
        try {
            // Deduct tokens from user's account
            $stmt_deduct = $conn->prepare("UPDATE users SET tokens = tokens - ? WHERE id = ? AND tokens >= ?");
            $stmt_deduct->bind_param("iii", $amount, $user_id, $amount);
            $stmt_deduct->execute();

            if ($stmt_deduct->affected_rows > 0) {
                // Insert transaction record
                $stmt_insert = $conn->prepare("INSERT INTO transactions (user_id, amount, payment_method, address, status) VALUES (?, ?, ?, ?, 'pending')");
                $stmt_insert->bind_param("isss", $user_id, $amount, $method, $address);
                $stmt_insert->execute();
                
                $conn->commit();
                $message = "Your payment request has been submitted successfully!";
                // Refresh token count
                $current_tokens -= $amount;
            } else {
                // Rollback if token deduction failed (e.g., insufficient funds race condition)
                $conn->rollback();
                $error = "Transaction failed. Please check your token balance and try again.";
            }
        } catch (Exception $e) {
            $conn->rollback();
            $error = "An error occurred. Please try again later.";
        }
    }
}

// Get user's transaction history
$transactions = [];
$stmt_history = $conn->prepare("SELECT amount, payment_method, address, status, created_at FROM transactions WHERE user_id = ? ORDER BY created_at DESC");
$stmt_history->bind_param("i", $user_id);
$stmt_history->execute();
$history_result = $stmt_history->get_result();
while ($row = $history_result->fetch_assoc()) {
    $transactions[] = $row;
}
$stmt_history->close();
$conn->close();

function getStatusBadge($status) {
    switch ($status) {
        case 'approved': return 'bg-green-100 text-green-800';
        case 'rejected': return 'bg-red-100 text-red-800';
        default: return 'bg-yellow-100 text-yellow-800';
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Request</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style> body { font-family: 'Inter', sans-serif; } </style>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto p-4 md:p-8 max-w-4xl">
        <div class="bg-white rounded-2xl shadow-xl p-8">
            <div class="flex justify-between items-center mb-6 border-b pb-4">
                <h1 class="text-3xl font-bold text-gray-800">Request a Payment</h1>
                <a href="index.php" class="text-indigo-600 hover:text-indigo-800 font-semibold">← Back to Dashboard</a>
            </div>

            <?php if ($message): ?>
            <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-6" role="alert"><p><?php echo $message; ?></p></div>
            <?php endif; ?>
            <?php if ($error): ?>
            <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-6" role="alert"><p><?php echo $error; ?></p></div>
            <?php endif; ?>
            
            <!-- Request Form -->
            <div class="bg-indigo-50 p-6 rounded-lg mb-8">
                <div class="flex justify-between items-center mb-4">
                     <h2 class="text-xl font-semibold text-gray-700">New Withdrawal Request</h2>
                     <p class="font-bold text-indigo-600 text-lg">Your Balance: <?php echo number_format($current_tokens); ?> Tokens</p>
                </div>
                <form method="POST" class="space-y-4">
                    <div>
                        <label for="amount" class="block text-sm font-medium text-gray-700">Amount of Tokens to Withdraw</label>
                        <input type="number" name="amount" id="amount" required class="mt-1 block w-full rounded-md border-gray-300 shadow-sm" placeholder="e.g., 1000">
                        <p class="text-xs text-gray-500 mt-1">Minimum withdrawal: <?php echo number_format($min_withdrawal_amount); ?> tokens.</p>
                    </div>
                     <div>
                        <label for="payment_method" class="block text-sm font-medium text-gray-700">Payment Method</label>
                        <select name="payment_method" id="payment_method" required class="mt-1 block w-full rounded-md border-gray-300 shadow-sm">
                            <option value="">-- Select a Method --</option>
                             <?php foreach ($payment_methods as $method_name): ?>
                                <option value="<?php echo htmlspecialchars($method_name); ?>"><?php echo htmlspecialchars($method_name); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div>
                        <label for="address" class="block text-sm font-medium text-gray-700">Your Payment Address</label>
                        <input type="text" name="address" id="address" required class="mt-1 block w-full rounded-md border-gray-300 shadow-sm" placeholder="e.g., Your Bkash number or wallet address">
                    </div>
                    <div>
                        <button type="submit" class="w-full bg-indigo-600 text-white font-semibold py-2.5 px-4 rounded-lg hover:bg-indigo-700">Submit Request</button>
                    </div>
                </form>
            </div>
            
            <!-- History Table -->
            <div>
                 <h2 class="text-xl font-semibold text-gray-700 mb-4">Your Payment History</h2>
                 <div class="overflow-x-auto rounded-lg border">
                     <table class="min-w-full divide-y divide-gray-200">
                         <thead class="bg-gray-50">
                             <tr>
                                 <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                 <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                                 <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Method</th>
                                 <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Address</th>
                                 <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                             </tr>
                         </thead>
                         <tbody class="bg-white divide-y divide-gray-200">
                             <?php if (empty($transactions)): ?>
                                 <tr><td colspan="5" class="px-6 py-4 text-center text-gray-500">No transaction history found.</td></tr>
                             <?php else: ?>
                                 <?php foreach ($transactions as $tx): ?>
                                     <tr>
                                         <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo date("d M, Y", strtotime($tx['created_at'])); ?></td>
                                         <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900"><?php echo number_format($tx['amount']); ?></td>
                                         <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo htmlspecialchars($tx['payment_method']); ?></td>
                                         <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 truncate" title="<?php echo htmlspecialchars($tx['address']); ?>"><?php echo htmlspecialchars($tx['address']); ?></td>
                                         <td class="px-6 py-4 whitespace-nowrap text-sm">
                                             <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full <?php echo getStatusBadge($tx['status']); ?>">
                                                 <?php echo ucfirst($tx['status']); ?>
                                             </span>
                                         </td>
                                     </tr>
                                 <?php endforeach; ?>
                             <?php endif; ?>
                         </tbody>
                     </table>
                 </div>
            </div>
        </div>
    </div>
</body>
</html>

