<?php
require_once 'db_connection.php';
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}
$user_id = $_SESSION['user_id'];

// Fetch settings for branding
$settings = [];
$result = $conn->query("SELECT setting_key, setting_value FROM settings");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
}
$site_name = htmlspecialchars($settings['site_name'] ?? 'DATLAB');
$site_favicon = htmlspecialchars($settings['site_favicon'] ?? '');

// Fetch posts for the current user
$posts = [];
$stmt = $conn->prepare("SELECT id, content, image_url, created_at FROM posts WHERE user_id = ? ORDER BY created_at DESC");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
while ($row = $result->fetch_assoc()) {
    $posts[] = $row;
}
$stmt->close();
$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Posts - <?php echo $site_name; ?></title>
    <?php if ($site_favicon): ?><link rel="icon" href="<?php echo $site_favicon; ?>"><?php endif; ?>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style> body { font-family: 'Inter', sans-serif; background-color: #f0f2f5; } </style>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto p-4 md:p-8 max-w-4xl">
        <div class="bg-white rounded-2xl shadow-xl p-8">
            <div class="flex justify-between items-center mb-6 border-b pb-4">
                <h1 class="text-3xl font-bold text-gray-800">My Posts</h1>
                <a href="index.php" class="text-indigo-600 hover:text-indigo-800 font-semibold">← Back to Feed</a>
            </div>

            <div id="posts-container" class="space-y-6">
                <?php if (empty($posts)): ?>
                    <p class="text-center text-gray-500 py-10">You haven't created any posts yet.</p>
                <?php else: ?>
                    <?php foreach ($posts as $post): ?>
                        <div class="bg-white p-5 rounded-lg border border-gray-200" id="post-<?php echo $post['id']; ?>">
                            <p class="text-gray-700 mb-4"><?php echo nl2br(htmlspecialchars($post['content'])); ?></p>
                            <?php if ($post['image_url']): ?>
                                <img src="<?php echo htmlspecialchars($post['image_url']); ?>" class="rounded-lg w-full object-cover max-h-80 mb-4">
                            <?php endif; ?>
                            <div class="flex justify-between items-center text-sm text-gray-500">
                                <span><?php echo date("F j, Y, g:i a", strtotime($post['created_at'])); ?></span>
                                <div class="space-x-4">
                                    <a href="edit_post.php?id=<?php echo $post['id']; ?>" class="font-semibold text-indigo-600 hover:text-indigo-800">Edit</a>
                                    <button onclick="deletePost(<?php echo $post['id']; ?>)" class="font-semibold text-red-600 hover:text-red-800">Delete</button>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <div id="toast" class="fixed bottom-5 right-5 bg-green-600 text-white py-2 px-4 rounded-lg shadow-lg opacity-0 transition-all duration-300"></div>

    <script>
        function showToast(message, isError = false) {
            const toast = document.getElementById('toast');
            toast.textContent = message;
            toast.className = `fixed bottom-5 right-5 text-white py-2 px-4 rounded-lg shadow-lg transition-all duration-300 ${isError ? 'bg-red-600' : 'bg-green-600'}`;
            toast.classList.remove('opacity-0');
            setTimeout(() => { toast.classList.add('opacity-0'); }, 3000);
        }

        async function deletePost(postId) {
            if (confirm('Are you sure you want to delete this post? This action cannot be undone.')) {
                try {
                    const response = await fetch(`index.php?action=delete_post&id=${postId}`);
                    const result = await response.json();
                    showToast(result.message, !result.success);
                    if (result.success) {
                        document.getElementById(`post-${postId}`).remove();
                    }
                } catch (error) {
                    showToast('An error occurred while deleting the post.', true);
                }
            }
        }
    </script>
</body>
</html>
