<?php
require_once 'db_connection.php';

// Check for referral code in URL for guests
if (!isset($_SESSION['user_id']) && isset($_GET['ref'])) {
    $_SESSION['referral_code_from_url'] = trim($_GET['ref']);
}

// সাইট-ব্যাপী সেটিংস লোড করুন
$settings = [];
$result = $conn->query("SELECT setting_key, setting_value FROM settings");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
}
$site_name = htmlspecialchars($settings['site_name'] ?? 'DATLAB');
$site_favicon = htmlspecialchars($settings['site_favicon'] ?? '');
$header_title = htmlspecialchars($settings['header_title'] ?? 'Feed');


// API অনুরোধ হ্যান্ডেল করার জন্য
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    $response = [];

    switch ($_POST['action']) {
        case 'register':
            $name = $_POST['name'] ?? '';
            $email = $_POST['email'] ?? '';
            $password = $_POST['password'] ?? '';
            $referral_code_input = trim($_POST['referral_code'] ?? '');
            $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'UNKNOWN';

            // Check if IP already exists
            $stmt_ip = $conn->prepare("SELECT id FROM users WHERE ip_address = ?");
            $stmt_ip->bind_param("s", $ip_address);
            $stmt_ip->execute();
            $stmt_ip->store_result();
            if ($stmt_ip->num_rows > 0) {
                echo json_encode(['success' => false, 'message' => 'An account has already been created from this device.']);
                $stmt_ip->close();
                $conn->close();
                exit();
            }
            $stmt_ip->close();

            if (empty($name) || empty($email) || empty($password)) {
                $response = ['success' => false, 'message' => 'Please fill all fields.'];
            } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $response = ['success' => false, 'message' => 'Invalid email format.'];
            } else {
                $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
                $stmt->bind_param("s", $email);
                $stmt->execute();
                $stmt->store_result();
                if ($stmt->num_rows > 0) {
                    $response = ['success' => false, 'message' => 'Email already exists.'];
                } else {
                    $referrer_id = null;
                    if (!empty($referral_code_input)) {
                        $stmt_ref = $conn->prepare("SELECT id FROM users WHERE referral_code = ?");
                        $stmt_ref->bind_param("s", $referral_code_input);
                        $stmt_ref->execute();
                        $result_ref = $stmt_ref->get_result();
                        if ($referrer = $result_ref->fetch_assoc()) {
                            $referrer_id = $referrer['id'];
                        }
                        $stmt_ref->close();
                    }
                    
                    $initial_tokens = 0;
                    if ($referrer_id) {
                        $initial_tokens = (int)($settings['new_user_referral_bonus'] ?? 100);
                    }

                    $hashed_password = password_hash($password, PASSWORD_BCRYPT);
                    $new_user_referral_code = uniqid('ref_');

                    $stmt_insert = $conn->prepare("INSERT INTO users (name, email, password, referral_code, referred_by, ip_address, tokens) VALUES (?, ?, ?, ?, ?, ?, ?)");
                    $stmt_insert->bind_param("ssssisi", $name, $email, $hashed_password, $new_user_referral_code, $referrer_id, $ip_address, $initial_tokens);
                    
                    if ($stmt_insert->execute()) {
                        if ($referrer_id) {
                            $referrer_bonus = (int)($settings['referrer_bonus'] ?? 50);
                            $stmt_bonus = $conn->prepare("UPDATE users SET tokens = tokens + ? WHERE id = ?");
                            $stmt_bonus->bind_param("ii", $referrer_bonus, $referrer_id);
                            $stmt_bonus->execute();
                            $stmt_bonus->close();
                        }
                        $response = ['success' => true, 'message' => 'Registration successful!'];
                    } else {
                        $response = ['success' => false, 'message' => 'Registration failed.'];
                    }
                    $stmt_insert->close();
                }
                $stmt->close();
            }
            break;

        case 'login':
            $email = $_POST['email'] ?? '';
            $password = $_POST['password'] ?? '';
            $stmt = $conn->prepare("SELECT id, name, password, tokens, last_claim_timestamp, profile_picture, is_admin, status FROM users WHERE email = ?");
            $stmt->bind_param("s", $email);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($user = $result->fetch_assoc()) {
                if (password_verify($password, $user['password'])) {
                    if ($user['status'] === 'blocked') {
                        $response = ['success' => false, 'message' => 'Your account has been blocked by an administrator.'];
                    } else {
                        $_SESSION['user_id'] = $user['id'];
                        if ($user['is_admin']) {
                            $_SESSION['is_admin'] = true;
                        }
                        unset($user['password']); // Remove password from response
                        $response = ['success' => true, 'user' => $user];
                    }
                } else {
                    $response = ['success' => false, 'message' => 'Incorrect password.'];
                }
            } else {
                $response = ['success' => false, 'message' => 'User not found.'];
            }
            $stmt->close();
            break;
            
        case 'create_post':
            if (!isset($_SESSION['user_id'])) {
                $response = ['success' => false, 'message' => 'You must be logged in to post.'];
                break;
            }
            $user_id = $_SESSION['user_id'];
            
            // Get user's package details
            $user_details_sql = "SELECT u.*, p.daily_post_limit, p.post_token_reward 
                                 FROM users u 
                                 LEFT JOIN packages p ON u.package_id = p.id 
                                 WHERE u.id = ?";
            $stmt_user = $conn->prepare($user_details_sql);
            $stmt_user->bind_param("i", $user_id);
            $stmt_user->execute();
            $current_user_details = $stmt_user->get_result()->fetch_assoc();
            $stmt_user->close();

            $daily_post_limit = (int)($current_user_details['daily_post_limit'] ?? $settings['daily_post_limit'] ?? 5);
            $post_reward = (int)($current_user_details['post_token_reward'] ?? $settings['post_token_reward'] ?? 5);

            // Check daily post limit
            $stmt_check = $conn->prepare("SELECT COUNT(*) as post_count FROM posts WHERE user_id = ? AND created_at >= CURDATE()");
            $stmt_check->bind_param("i", $user_id);
            $stmt_check->execute();
            $post_count = $stmt_check->get_result()->fetch_assoc()['post_count'];
            $stmt_check->close();

            if ($post_count >= $daily_post_limit) {
                 $response = ['success' => false, 'message' => "You have reached your daily post limit of {$daily_post_limit} posts."];
                 break;
            }

            $content = $_POST['content'] ?? '';
            $image_url = null;

            if (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
                $target_dir = "uploads/";
                if (!is_dir($target_dir)) { mkdir($target_dir, 0777, true); }
                $image_name = time() . '_' . basename($_FILES["image"]["name"]);
                $target_file = $target_dir . $image_name;
                if (move_uploaded_file($_FILES["image"]["tmp_name"], $target_file)) {
                    $image_url = $target_file;
                }
            }
            
            $stmt = $conn->prepare("INSERT INTO posts (user_id, content, image_url) VALUES (?, ?, ?)");
            $stmt->bind_param("iss", $user_id, $content, $image_url);
            if ($stmt->execute()) {
                 $stmt_update = $conn->prepare("UPDATE users SET tokens = tokens + ? WHERE id = ?");
                 $stmt_update->bind_param("ii", $post_reward, $user_id);
                 $stmt_update->execute();
                 $stmt_update->close();

                 $stmt_get_tokens = $conn->prepare("SELECT tokens FROM users WHERE id = ?");
                 $stmt_get_tokens->bind_param("i", $user_id);
                 $stmt_get_tokens->execute();
                 $user_data = $stmt_get_tokens->get_result()->fetch_assoc();
                 $new_token_balance = $user_data['tokens'];
                 $stmt_get_tokens->close();

                 $response = ['success' => true, 'message' => "Post created! You earned {$post_reward} tokens.", 'new_tokens' => $new_token_balance];
            } else {
                $response = ['success' => false, 'message' => 'Failed to create post.'];
            }
            $stmt->close();
            break;

        case 'claim_token':
            if (!isset($_SESSION['user_id'])) {
                $response = ['success' => false, 'message' => 'You must be logged in.'];
                break;
            }
            $user_id = $_SESSION['user_id'];
            $claim_reward = (int)($settings['claim_now_reward'] ?? 50);

            $stmt = $conn->prepare("SELECT tokens, last_claim_timestamp FROM users WHERE id = ?");
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $user = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            $now = round(microtime(true) * 1000);
            $twentyFourHours = 24 * 60 * 60 * 1000;

            if ($now - $user['last_claim_timestamp'] >= $twentyFourHours) {
                $new_tokens = $user['tokens'] + $claim_reward;
                $stmt_update = $conn->prepare("UPDATE users SET tokens = ?, last_claim_timestamp = ? WHERE id = ?");
                $stmt_update->bind_param("idi", $new_tokens, $now, $user_id);
                if ($stmt_update->execute()) {
                    $response = ['success' => true, 'message' => "You have successfully claimed {$claim_reward} tokens!", 'tokens' => $new_tokens, 'last_claim' => $now];
                } else {
                    $response = ['success' => false, 'message' => 'Could not update tokens.'];
                }
                $stmt_update->close();
            } else {
                 $remainingTime = $twentyFourHours - ($now - $user['last_claim_timestamp']);
                 $hours = floor($remainingTime / (1000 * 60 * 60));
                 $minutes = floor(($remainingTime % (1000 * 60 * 60)) / (1000 * 60));
                 $response = ['success' => false, 'message' => "You can claim again in {$hours}h {$minutes}m."];
            }
            break;
        
        case 'add_comment':
            if (!isset($_SESSION['user_id'])) {
                $response = ['success' => false, 'message' => 'You must be logged in to comment.'];
                break;
            }
            $user_id = $_SESSION['user_id'];
             
            $user_details_sql = "SELECT u.*, p.daily_comment_limit, p.comment_token_reward 
                                 FROM users u 
                                 LEFT JOIN packages p ON u.package_id = p.id 
                                 WHERE u.id = ?";
            $stmt_user = $conn->prepare($user_details_sql);
            $stmt_user->bind_param("i", $user_id);
            $stmt_user->execute();
            $current_user_details = $stmt_user->get_result()->fetch_assoc();
            $stmt_user->close();

            $daily_comment_limit = (int)($current_user_details['daily_comment_limit'] ?? $settings['daily_comment_limit'] ?? 20);
            $comment_reward = (int)($current_user_details['comment_token_reward'] ?? $settings['comment_token_reward'] ?? 2);

            $stmt_check = $conn->prepare("SELECT COUNT(*) as comment_count FROM comments WHERE user_id = ? AND created_at >= CURDATE()");
            $stmt_check->bind_param("i", $user_id);
            $stmt_check->execute();
            $comment_count = $stmt_check->get_result()->fetch_assoc()['comment_count'];
            $stmt_check->close();

            if ($comment_count >= $daily_comment_limit) {
                $response = ['success' => false, 'message' => "You have reached your daily comment limit of {$daily_comment_limit}."];
                break;
            }

            $post_id = $_POST['post_id'] ?? 0;
            $content = $_POST['content'] ?? '';

            if (empty($content) || empty($post_id)) {
                $response = ['success' => false, 'message' => 'Comment cannot be empty.'];
                break;
            }

            $stmt_post_owner = $conn->prepare("SELECT user_id FROM posts WHERE id = ?");
            $stmt_post_owner->bind_param("i", $post_id);
            $stmt_post_owner->execute();
            $post_owner_id = $stmt_post_owner->get_result()->fetch_assoc()['user_id'];
            $stmt_post_owner->close();

            $stmt_insert = $conn->prepare("INSERT INTO comments (post_id, user_id, content) VALUES (?, ?, ?)");
            $stmt_insert->bind_param("iis", $post_id, $user_id, $content);
            
            if ($stmt_insert->execute()) {
                $award_tokens = false;
                if ($user_id != $post_owner_id) {
                    $stmt_check_first = $conn->prepare("SELECT COUNT(*) as comment_count FROM comments WHERE user_id = ? AND post_id = ?");
                    $stmt_check_first->bind_param("ii", $user_id, $post_id);
                    $stmt_check_first->execute();
                    $is_first_comment = $stmt_check_first->get_result()->fetch_assoc()['comment_count'];
                    $stmt_check_first->close();

                    if ($is_first_comment == 1) {
                        $award_tokens = true;
                        $stmt_update = $conn->prepare("UPDATE users SET tokens = tokens + ? WHERE id = ?");
                        $stmt_update->bind_param("ii", $comment_reward, $user_id);
                        $stmt_update->execute();
                        $stmt_update->close();
                    }
                }

                $stmt_get_tokens = $conn->prepare("SELECT tokens FROM users WHERE id = ?");
                $stmt_get_tokens->bind_param("i", $user_id);
                $stmt_get_tokens->execute();
                $new_token_balance = $stmt_get_tokens->get_result()->fetch_assoc()['tokens'];
                $stmt_get_tokens->close();

                $message = 'Comment added successfully.';
                if ($award_tokens) {
                    $message .= " You earned {$comment_reward} tokens.";
                }
                $response = ['success' => true, 'message' => $message, 'new_tokens' => $new_token_balance];
            } else {
                $response = ['success' => false, 'message' => 'Failed to add comment.'];
            }
            $stmt_insert->close();
            break;
        case 'mark_notifications_read':
            if (!isset($_SESSION['user_id'])) {
                $response = ['success' => false, 'message' => 'Unauthorized'];
                break;
            }
            $user_id = $_SESSION['user_id'];
            
            $sql_unread = "SELECT id FROM notifications WHERE id NOT IN (SELECT notification_id FROM user_notifications WHERE user_id = ?)";
            $stmt_unread = $conn->prepare($sql_unread);
            $stmt_unread->bind_param("i", $user_id);
            $stmt_unread->execute();
            $result_unread = $stmt_unread->get_result();
            $unread_ids = [];
            while ($row = $result_unread->fetch_assoc()) {
                $unread_ids[] = $row['id'];
            }
            $stmt_unread->close();

            if (!empty($unread_ids)) {
                $sql_insert = "INSERT INTO user_notifications (user_id, notification_id) VALUES ";
                $params = [];
                $types = "";
                foreach ($unread_ids as $notif_id) {
                    $sql_insert .= "(?, ?),";
                    $params[] = $user_id;
                    $params[] = $notif_id;
                    $types .= "ii";
                }
                $sql_insert = rtrim($sql_insert, ',');
                $sql_insert .= " ON DUPLICATE KEY UPDATE is_read=1";
                $stmt_insert = $conn->prepare($sql_insert);
                $stmt_insert->bind_param($types, ...$params);
                $stmt_insert->execute();
                $stmt_insert->close();
            }
            $response = ['success' => true];
            break;
    }
    echo json_encode($response);
    $conn->close();
    exit();
}

if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    if (isset($_GET['action'])) {
        header('Content-Type: application/json');
        if ($_GET['action'] == 'logout') {
            session_destroy();
            echo json_encode(['success' => true]);
            exit();
        }
        if ($_GET['action'] == 'get_posts') {
            $sql = "SELECT p.id, p.content, p.image_url, p.created_at, u.name as user_name, u.profile_picture as user_avatar FROM posts p JOIN users u ON p.user_id = u.id ORDER BY p.created_at DESC";
            $result = $conn->query($sql);
            $posts = [];
            while($row = $result->fetch_assoc()) {
                $posts[] = $row;
            }
            echo json_encode($posts);
            $conn->close();
            exit();
        }
        if ($_GET['action'] == 'get_comments') {
            $post_id = $_GET['post_id'] ?? 0;
            $sql = "SELECT c.id, c.content, c.created_at, u.name as user_name, u.profile_picture as user_avatar FROM comments c JOIN users u ON c.user_id = u.id WHERE c.post_id = ? ORDER BY c.created_at ASC";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("i", $post_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $comments = [];
            while($row = $result->fetch_assoc()) {
                $comments[] = $row;
            }
            echo json_encode($comments);
            $conn->close();
            exit();
        }
        if ($_GET['action'] == 'get_notifications') {
            if (!isset($_SESSION['user_id'])) {
                echo json_encode(['success' => false, 'message' => 'Unauthorized']);
                exit();
            }
            $user_id = $_SESSION['user_id'];
            $sql = "SELECT n.id, n.title, n.content, n.image_url, n.link_url, n.created_at FROM notifications n WHERE n.id NOT IN (SELECT notification_id FROM user_notifications WHERE user_id = ?) ORDER BY n.created_at DESC LIMIT 5";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $notifications = [];
            while($row = $result->fetch_assoc()) {
                $notifications[] = $row;
            }
            echo json_encode(['success' => true, 'notifications' => $notifications]);
            $conn->close();
            exit();
        }
    }
}

// ব্যবহারকারী লগইন করা থাকলে তার তথ্য লোড করুন
$currentUser = null;
if (isset($_SESSION['user_id'])) {
    $stmt = $conn->prepare("SELECT u.*, p.name as package_name 
                            FROM users u 
                            LEFT JOIN packages p ON u.package_id = p.id 
                            WHERE u.id = ?");
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $currentUser = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    
    // অপঠিত নোটিফিকেশনের সংখ্যা গণনা করুন
    if ($currentUser) {
        $stmt_count = $conn->prepare("SELECT COUNT(n.id) as count FROM notifications n WHERE n.id NOT IN (SELECT notification_id FROM user_notifications WHERE user_id = ?)");
        $stmt_count->bind_param("i", $_SESSION['user_id']);
        $stmt_count->execute();
        $unread_count = $stmt_count->get_result()->fetch_assoc()['count'];
        $currentUser['unread_notifications'] = $unread_count;
        $stmt_count->close();
    }
}
$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $site_name; ?></title>
    <?php if ($site_favicon): ?>
    <link rel="icon" href="<?php echo $site_favicon; ?>">
    <?php endif; ?>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Inter', sans-serif; background-color: #f0f2f5; }
        .nav-link.active { background-color: #eef2ff; color: #4f46e5; font-weight: 600; }
        .nav-link:hover { background-color: #f3f4f6; }
    </style>
    <?php if (!empty($settings['ad_code_header'])): ?>
    <?php echo $settings['ad_code_header']; ?>
    <?php endif; ?>
</head>
<body class="bg-gray-100">

    <div id="auth-modal" class="hidden fixed inset-0 bg-gray-900 bg-opacity-75 flex items-center justify-center z-50 p-4">
        <div class="bg-white rounded-lg shadow-xl w-full max-w-md relative">
            <button id="close-modal-btn" class="absolute top-3 right-3 text-gray-500 hover:text-gray-800">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>
            </button>
            <div class="p-8">
                <div id="login-form">
                    <h2 class="text-center text-3xl font-extrabold text-gray-900 mb-8">Sign in</h2>
                    <form id="login-form-tag" class="space-y-6">
                        <input id="login-email" name="email" type="email" required class="w-full px-3 py-2 border border-gray-300 rounded-md" placeholder="Email address">
                        <input id="login-password" name="password" type="password" required class="w-full px-3 py-2 border border-gray-300 rounded-md" placeholder="Password">
                        <button type="submit" id="login-btn" class="w-full py-2 px-4 rounded-md text-white bg-indigo-600 hover:bg-indigo-700">Sign in</button>
                    </form>
                    <p class="mt-4 text-center text-sm">Don't have an account? <a href="#" id="show-register" class="font-medium text-indigo-600">Sign up</a></p>
                </div>
                <div id="register-form" class="hidden">
                    <h2 class="text-center text-3xl font-extrabold text-gray-900 mb-8">Create an account</h2>
                    <form id="register-form-tag" class="space-y-6">
                        <input id="register-name" name="name" type="text" required class="w-full px-3 py-2 border border-gray-300 rounded-md" placeholder="Full Name">
                        <input id="register-email" name="email" type="email" required class="w-full px-3 py-2 border border-gray-300 rounded-md" placeholder="Email address">
                        <input id="register-password" name="password" type="password" required class="w-full px-3 py-2 border border-gray-300 rounded-md" placeholder="Password">
                        <input id="referral-code" name="referral_code" type="text" class="w-full px-3 py-2 border border-gray-300 rounded-md" placeholder="Referral Code (Optional)">
                        <button type="submit" id="register-btn" class="w-full py-2 px-4 rounded-md text-white bg-indigo-600 hover:bg-indigo-700">Sign up</button>
                    </form>
                    <p class="mt-4 text-center text-sm">Already have an account? <a href="#" id="show-login" class="font-medium text-indigo-600">Sign in</a></p>
                </div>
            </div>
        </div>
    </div>

    <div id="dashboard-container" class="relative min-h-screen md:flex">
        <div id="sidebar-overlay" class="fixed inset-0 bg-black bg-opacity-50 z-30 hidden md:hidden"></div>
        <aside id="sidebar" class="fixed inset-y-0 left-0 bg-white shadow-lg w-64 z-40 transform -translate-x-full md:relative md:translate-x-0 transition-transform duration-300 ease-in-out flex flex-col">
            <div class="p-5 border-b flex items-center justify-between space-x-2">
                <div class="flex items-center space-x-2">
                    <?php if (!empty($settings['site_logo'])): ?>
                        <img src="<?php echo htmlspecialchars($settings['site_logo']); ?>" alt="Logo" class="h-8 max-w-[150px] object-contain">
                    <?php else: ?>
                        <i class="fas fa-users text-2xl text-indigo-600"></i>
                        <h1 class="text-xl font-bold text-gray-800"><?php echo $site_name; ?></h1>
                    <?php endif; ?>
                </div>
                <button id="menu-close-btn" class="md:hidden text-gray-500 hover:text-gray-800">
                     <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>
                </button>
            </div>
            <div class="flex flex-col justify-between flex-1 overflow-y-auto">
                <nav class="px-4 py-6 space-y-2">
                    <div id="user-controls" class="hidden">
                        <div class="flex items-center space-x-4 mb-6 px-2">
                            <img src="https://placehold.co/48x48/e2e8f0/64748b?text=U" id="user-avatar-sidebar" alt="Profile Picture" class="w-12 h-12 rounded-full object-cover border-2 border-indigo-500">
                            <div>
                                <p id="user-name-sidebar" class="font-semibold text-gray-800"></p>
                                <p class="text-sm text-yellow-500 font-semibold mt-1">Tokens: <span id="token-balance">0</span></p>
                            </div>
                        </div>
                    </div>
                    <div id="guest-controls">
                         <div class="flex items-center space-x-4 mb-6 px-2">
                            <div class="w-12 h-12 rounded-full bg-gray-200 flex items-center justify-center">
                               <svg class="w-8 h-8 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path></svg>
                            </div>
                            <div>
                                <p class="font-semibold text-gray-800">Guest</p>
                                <button class="login-prompt-btn text-sm text-indigo-600 hover:underline">Login to continue</button>
                            </div>
                        </div>
                    </div>
                    <a href="index.php" class="nav-link active flex items-center px-4 py-2.5 text-gray-700 rounded-lg">
                        <i class="fas fa-stream w-5 text-center mr-3"></i><span>Feed</span>
                    </a>
                    <div id="user-nav-links" class="hidden space-y-2">
                        <?php if(($settings['show_profile'] ?? '1') == '1'): ?>
                        <a href="editprofile.php" class="nav-link flex items-center px-4 py-2.5 text-gray-600 rounded-lg">
                            <i class="fas fa-user w-5 text-center mr-3"></i><span>Profile</span>
                        </a>
                        <?php endif; ?>

                        <a href="myposts.php" class="nav-link flex items-center px-4 py-2.5 text-gray-600 rounded-lg">
                            <i class="fas fa-copy w-5 text-center mr-3"></i><span>My Posts</span>
                        </a>
                        
                        <?php if(($settings['show_my_referrals'] ?? '1') == '1'): ?>
                        <a href="myreferrals.php" class="nav-link flex items-center px-4 py-2.5 text-gray-600 rounded-lg">
                            <i class="fas fa-users w-5 text-center mr-3"></i><span>My Referrals</span>
                        </a>
                        <?php endif; ?>
                        
                        <?php if(($settings['show_buy_package'] ?? '1') == '1'): ?>
                        <a href="buypackage.php" class="nav-link flex items-center px-4 py-2.5 text-gray-600 rounded-lg">
                            <i class="fas fa-shopping-bag w-5 text-center mr-3"></i><span>Buy Package</span>
                        </a>
                        <?php endif; ?>

                        <?php if(($settings['show_notifications'] ?? '1') == '1'): ?>
                        <a href="notifications.php" class="nav-link flex items-center px-4 py-2.5 text-gray-600 rounded-lg">
                            <i class="fas fa-bell w-5 text-center mr-3"></i><span>Notifications</span>
                        </a>
                        <?php endif; ?>
                        
                        <?php if(($settings['show_support'] ?? '1') == '1'): ?>
                        <a href="support.php" class="nav-link flex items-center px-4 py-2.5 text-gray-600 rounded-lg">
                            <i class="fas fa-life-ring w-5 text-center mr-3"></i><span>Support</span>
                        </a>
                        <?php endif; ?>

                        <?php if(($settings['show_payment'] ?? '1') == '1'): ?>
                        <a href="payment.php" class="nav-link flex items-center px-4 py-2.5 text-gray-600 rounded-lg">
                            <i class="fas fa-credit-card w-5 text-center mr-3"></i><span>Payment</span>
                        </a>
                        <?php endif; ?>
                        <?php if(($settings['show_payment_proofs'] ?? '1') == '1'): ?>
                        <a href="paymentproof.php" class="nav-link flex items-center px-4 py-2.5 text-gray-600 rounded-lg">
                           <i class="fas fa-check-circle w-5 text-center mr-3"></i><span>Payment Proofs</span>
                        </a>
                        <?php endif; ?>
                        <a href="#" id="logout-btn" class="nav-link flex items-center px-4 py-2.5 text-gray-600 rounded-lg">
                            <i class="fas fa-sign-out-alt w-5 text-center mr-3"></i><span>Logout</span>
                        </a>
                        <div class="pt-4">
                            <button id="claim-now-btn" class="w-full bg-indigo-600 text-white font-semibold py-2.5 px-4 rounded-lg hover:bg-indigo-700 transition-colors">
                                <i class="fas fa-gift mr-2"></i>Claim Now
                            </button>
                        </div>
                    </div>
                </nav>
            </div>
        </aside>

        <main class="flex-1 overflow-y-auto">
            <header class="bg-white shadow-sm p-4 flex justify-between items-center sticky top-0 z-20">
                <button id="menu-open-btn" class="md:hidden text-gray-600 hover:text-gray-800">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path></svg>
                </button>
                <h2 class="text-xl font-semibold text-gray-800 hidden md:block"><?php echo $header_title; ?></h2>
                <h2 class="text-xl font-semibold text-gray-800 md:hidden"><?php echo $site_name; ?></h2>
                <div class="flex items-center space-x-4">
                    <div id="notification-bell-container" class="relative hidden">
                        <button id="notification-bell" class="relative text-gray-600 hover:text-gray-800 focus:outline-none">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9"></path></svg>
                            <span id="notification-count" class="absolute -top-1 -right-1 bg-red-500 text-white text-xs w-4 h-4 rounded-full flex items-center justify-center" style="display: none;">0</span>
                        </button>
                        <div id="notification-dropdown" class="hidden absolute right-0 mt-2 w-80 bg-white rounded-lg shadow-xl z-30 overflow-hidden border">
                            <div class="p-4 font-semibold border-b text-gray-800">Notifications</div>
                            <div id="notification-list" class="max-h-96 overflow-y-auto"></div>
                            <a href="notifications.php" class="block text-center p-2 bg-gray-50 hover:bg-gray-100 text-sm font-medium text-indigo-600">
                                View All Notifications
                            </a>
                        </div>
                    </div>
                    <div id="header-user-controls" class="hidden">
                        <img src="https://placehold.co/40x40/e2e8f0/64748b?text=U" id="user-avatar-header" alt="Profile Picture" class="w-10 h-10 rounded-full object-cover">
                    </div>
                    <div id="header-guest-controls">
                        <button class="login-prompt-btn bg-indigo-600 text-white font-semibold py-2 px-4 rounded-lg text-sm">Login / Register</button>
                    </div>
                </div>
            </header>
            <div class="p-4 md:p-8">
                <div id="create-post-user" class="hidden bg-white p-6 rounded-lg shadow-md mb-8">
                    <h3 class="text-lg font-semibold mb-4">Create a new post</h3>
                    <form id="create-post-form">
                        <textarea name="content" id="postContent" class="w-full p-3 border rounded-lg" rows="4" placeholder="What's on your mind?" required></textarea>
                        <div class="mt-4">
                            <label for="postImage" class="block text-sm font-medium mb-2">Add a picture</label>
                            <input type="file" name="image" id="postImage" accept="image/*" class="w-full text-sm file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:font-semibold file:bg-indigo-50 file:text-indigo-700 hover:file:bg-indigo-100"/>
                            <img id="imagePreview" src="" class="mt-4 rounded-lg hidden max-h-48"/>
                        </div>
                        <div class="flex justify-end mt-4">
                            <button type="submit" class="bg-indigo-600 text-white font-semibold py-2 px-6 rounded-lg">Post</button>
                        </div>
                    </form>
                </div>
                <div id="create-post-guest" class="bg-white p-6 rounded-lg shadow-md mb-8 text-center">
                     <h3 class="text-lg font-semibold mb-2">Join the conversation!</h3>
                     <p class="text-gray-600 mb-4">Please log in to create your own posts.</p>
                     <button class="login-prompt-btn bg-indigo-600 text-white font-semibold py-2 px-6 rounded-lg">Login to Post</button>
                </div>
                <div id="postsContainer" class="space-y-6"></div>
            </div>
        </main>
    </div>

    <div id="toast" class="fixed bottom-5 right-5 bg-green-600 text-white py-2 px-4 rounded-lg shadow-lg opacity-0 transition-all duration-300"></div>

    <script>
        const currentUser = <?php echo json_encode($currentUser, JSON_NUMERIC_CHECK); ?>;
        const adSettings = {
            postsPerAd: <?php echo (int)($settings['posts_per_ad'] ?? 0); ?>,
            bodyCode: <?php echo json_encode($settings['ad_code_body'] ?? ''); ?>
        };
        const referralCodeFromUrl = '<?php echo isset($_SESSION['referral_code_from_url']) ? htmlspecialchars($_SESSION['referral_code_from_url']) : ''; ?>';
        <?php unset($_SESSION['referral_code_from_url']); ?>

        document.addEventListener('DOMContentLoaded', function() {
            const authModal = document.getElementById('auth-modal');
            const loginForm = document.getElementById('login-form');
            const registerForm = document.getElementById('register-form');
            const showRegisterLink = document.getElementById('show-register');
            const showLoginLink = document.getElementById('show-login');
            const closeModalBtn = document.getElementById('close-modal-btn');
            const loginPromptBtns = document.querySelectorAll('.login-prompt-btn');
            const toast = document.getElementById('toast');
            
            const notificationBellContainer = document.getElementById('notification-bell-container');
            const notificationBell = document.getElementById('notification-bell');
            const notificationCount = document.getElementById('notification-count');
            const notificationDropdown = document.getElementById('notification-dropdown');
            const notificationList = document.getElementById('notification-list');

            // Mobile Menu Toggle
            const sidebar = document.getElementById('sidebar');
            const overlay = document.getElementById('sidebar-overlay');
            const openBtn = document.getElementById('menu-open-btn');
            const closeBtn = document.getElementById('menu-close-btn');

            function toggleSidebar() {
                sidebar.classList.toggle('-translate-x-full');
                if (overlay) overlay.classList.toggle('hidden');
            }

            if(openBtn) openBtn.addEventListener('click', toggleSidebar);
            if(closeBtn) closeBtn.addEventListener('click', toggleSidebar);
            if(overlay) overlay.addEventListener('click', toggleSidebar);


            if (referralCodeFromUrl) {
                authModal.classList.remove('hidden');
                loginForm.style.display = 'none';
                registerForm.style.display = 'block';
                document.getElementById('referral-code').value = referralCodeFromUrl;
            }


            function escapeHTML(str) {
                if (str === null || str === undefined) return '';
                return str.replace(/[&<>'"]/g, 
                  tag => ({'&': '&amp;','<': '&lt;','>': '&gt;',"'": '&#39;','"': '&quot;'}[tag] || tag)
                );
            }

            function showToast(message, isError = false) {
                toast.textContent = message;
                toast.className = `fixed bottom-5 right-5 text-white py-2 px-4 rounded-lg shadow-lg transition-all duration-300 ${isError ? 'bg-red-600' : 'bg-green-600'}`;
                toast.classList.remove('opacity-0');
                setTimeout(() => { toast.classList.add('opacity-0'); }, 3000);
            }

            function updateUI(user) {
                const isLoggedIn = !!user;
                document.getElementById('guest-controls').style.display = isLoggedIn ? 'none' : 'block';
                document.getElementById('user-controls').style.display = isLoggedIn ? 'block' : 'none';
                document.getElementById('header-guest-controls').style.display = isLoggedIn ? 'none' : 'block';
                document.getElementById('header-user-controls').style.display = isLoggedIn ? 'block' : 'none';
                document.getElementById('create-post-guest').style.display = isLoggedIn ? 'none' : 'block';
                document.getElementById('create-post-user').style.display = isLoggedIn ? 'block' : 'none';
                document.getElementById('user-nav-links').style.display = isLoggedIn ? 'block' : 'none';

                if (isLoggedIn) {
                    const initial = escapeHTML(user.name.charAt(0).toUpperCase());
                    const avatarSidebar = document.getElementById('user-avatar-sidebar');
                    const avatarHeader = document.getElementById('user-avatar-header');
                    
                    avatarSidebar.src = user.profile_picture ? escapeHTML(user.profile_picture) : `https://placehold.co/48x48/6366f1/ffffff?text=${initial}`;
                    avatarHeader.src = user.profile_picture ? escapeHTML(user.profile_picture) : `https://placehold.co/40x40/6366f1/ffffff?text=${initial}`;

                    document.getElementById('user-name-sidebar').textContent = user.name;
                    document.getElementById('token-balance').textContent = user.tokens;
                    
                    notificationBellContainer.classList.remove('hidden');
                    const unreadCount = user.unread_notifications || 0;
                    notificationCount.textContent = unreadCount;
                    notificationCount.style.display = unreadCount > 0 ? 'flex' : 'none';
                } else {
                     notificationBellContainer.classList.add('hidden');
                }
            }
            
            function getAvatarUrl(user, size = 40) {
                 const initial = escapeHTML(user.user_name.charAt(0).toUpperCase());
                 const avatarSrc = user.user_avatar ? escapeHTML(user.user_avatar) : `https://placehold.co/${size}x${size}/f97316/ffffff?text=${initial}`;
                 return avatarSrc;
            }

            async function fetchPosts() {
                const postsContainer = document.getElementById('postsContainer');
                postsContainer.innerHTML = `<p class="text-center text-gray-500 py-10">Loading posts...</p>`;
                try {
                    const response = await fetch('index.php?action=get_posts');
                    const posts = await response.json();
                    
                    if (posts.length === 0) {
                        postsContainer.innerHTML = `<p class="text-center text-gray-500 py-10">No posts yet. Be the first to share something!</p>`;
                        return;
                    }
                    
                    postsContainer.innerHTML = '';
                    let postCounter = 0;
                    posts.forEach(post => {
                        const postElement = createPostElement(post);
                        postsContainer.appendChild(postElement);
                        fetchComments(post.id);

                        postCounter++;

                        if (adSettings.postsPerAd > 0 && adSettings.bodyCode.trim() !== '' && postCounter % adSettings.postsPerAd === 0) {
                            const adElement = document.createElement('div');
                            adElement.className = 'my-6 p-4 bg-gray-200 rounded-lg';
                            adElement.innerHTML = adSettings.bodyCode;
                            postsContainer.appendChild(adElement);
                        }
                    });
                } catch (error) {
                    postsContainer.innerHTML = `<p class="text-center text-red-500 py-10">Failed to load posts. Please try again later.</p>`;
                }
            }
            
            async function fetchComments(postId) {
                const commentsContainer = document.getElementById(`comments-container-${postId}`);
                const commentFormContainer = document.getElementById(`comment-form-container-${postId}`);
                
                try {
                    const response = await fetch(`index.php?action=get_comments&post_id=${postId}`);
                    const comments = await response.json();
                    commentsContainer.innerHTML = '';

                    comments.forEach(comment => {
                        const commentEl = document.createElement('div');
                        commentEl.className = 'flex items-start space-x-3';
                        commentEl.innerHTML = `
                            <img src="${getAvatarUrl(comment, 32)}" class="w-8 h-8 rounded-full object-cover">
                            <div class="bg-gray-100 rounded-lg p-3 flex-1">
                                <p class="font-semibold text-sm text-gray-800">${escapeHTML(comment.user_name)}</p>
                                <p class="text-sm text-gray-600">${escapeHTML(comment.content)}</p>
                            </div>
                        `;
                        commentsContainer.appendChild(commentEl);
                    });
                } catch (error) {
                    commentsContainer.innerHTML = `<p class="text-xs text-red-500">Could not load comments.</p>`;
                }

                if (currentUser) {
                    commentFormContainer.innerHTML = `
                        <form class="comment-form flex items-center space-x-2" data-post-id="${postId}">
                            <textarea class="comment-box w-full p-2 border rounded-lg text-sm" rows="1" placeholder="Write a comment..."></textarea>
                            <button type="submit" class="bg-indigo-500 text-white font-semibold py-2 px-4 rounded-lg text-sm hover:bg-indigo-600 transition-colors">Post</button>
                        </form>
                    `;
                } else {
                    commentFormContainer.innerHTML = `<button class="login-prompt-btn text-indigo-600 hover:underline text-sm font-semibold">Login to Comment</button>`;
                }
                commentFormContainer.querySelectorAll('.login-prompt-btn').forEach(btn => btn.addEventListener('click', () => authModal.classList.remove('hidden')));
            }

            function timeAgo(dateString) {
                const date = new Date(dateString.replace(' ', 'T')+'Z');
                const seconds = Math.floor((new Date() - date) / 1000);
                let interval = seconds / 31536000;
                if (interval > 1) return Math.floor(interval) + " years ago";
                interval = seconds / 2592000;
                if (interval > 1) return Math.floor(interval) + " months ago";
                interval = seconds / 86400;
                if (interval > 1) return Math.floor(interval) + " days ago";
                interval = seconds / 3600;
                if (interval > 1) return Math.floor(interval) + " hours ago";
                interval = seconds / 60;
                if (interval > 1) return Math.floor(interval) + " minutes ago";
                return "Just now";
            }
            
            function createPostElement(post) {
                const element = document.createElement('div');
                element.className = 'bg-white p-6 rounded-lg shadow-md';
                const imageHTML = post.image_url ? `<img src="${escapeHTML(post.image_url)}" class="mt-4 rounded-lg w-full object-cover max-h-96">` : '';
                element.innerHTML = `
                    <div class="flex items-center mb-4">
                        <img src="${getAvatarUrl(post)}" class="w-10 h-10 rounded-full object-cover mr-4">
                        <div>
                            <p class="font-semibold text-gray-800">${escapeHTML(post.user_name)}</p>
                            <p class="text-sm text-gray-500">${timeAgo(post.created_at)}</p>
                        </div>
                    </div>
                    <p class="text-gray-700">${escapeHTML(post.content).replace(/\n/g, '<br>')}</p>
                    ${imageHTML}
                    <div class="border-t mt-4 pt-4 space-y-4">
                        <div id="comments-container-${post.id}" class="space-y-3"></div>
                        <div id="comment-form-container-${post.id}"></div>
                    </div>
                `;
                return element;
            }
            
            loginPromptBtns.forEach(btn => btn.addEventListener('click', () => authModal.classList.remove('hidden')));
            closeModalBtn.addEventListener('click', () => authModal.classList.add('hidden'));
            showRegisterLink.addEventListener('click', e => { e.preventDefault(); loginForm.style.display = 'none'; registerForm.style.display = 'block'; });
            showLoginLink.addEventListener('click', e => { e.preventDefault(); registerForm.style.display = 'none'; loginForm.style.display = 'block'; });

            document.getElementById('register-form-tag').addEventListener('submit', async function(e) {
                e.preventDefault();
                const formData = new FormData(this);
                formData.append('action', 'register');
                const response = await fetch('index.php', { method: 'POST', body: formData });
                const result = await response.json();
                showToast(result.message, !result.success);
                if (result.success) {
                    registerForm.style.display = 'none';
                    loginForm.style.display = 'block';
                    this.reset();
                }
            });

            document.getElementById('login-form-tag').addEventListener('submit', async function(e) {
                e.preventDefault();
                const formData = new FormData(this);
                formData.append('action', 'login');
                const response = await fetch('index.php', { method: 'POST', body: formData });
                const result = await response.json();
                if (result.success) {
                    if (result.user && result.user.is_admin) {
                        window.location.href = 'admin/admindashboard.php';
                    } else {
                       location.reload();
                    }
                } else {
                    showToast(result.message, !result.success);
                }
            });

            document.getElementById('logout-btn').addEventListener('click', async (e) => {
                e.preventDefault();
                await fetch('index.php?action=logout');
                window.location.href = 'index.php';
            });

            document.getElementById('create-post-form').addEventListener('submit', async function(e) {
                e.preventDefault();
                const formData = new FormData(this);
                formData.append('action', 'create_post');
                const response = await fetch('index.php', { method: 'POST', body: formData });
                const result = await response.json();
                showToast(result.message, !result.success);
                if (result.success) {
                    this.reset();
                    document.getElementById('imagePreview').classList.add('hidden');
                    document.getElementById('token-balance').textContent = result.new_tokens;
                    fetchPosts();
                }
            });

            document.getElementById('postImage').addEventListener('change', function() {
                const preview = document.getElementById('imagePreview');
                if (this.files && this.files[0]) {
                    const reader = new FileReader();
                    reader.onload = e => { preview.src = e.target.result; preview.classList.remove('hidden'); }
                    reader.readAsDataURL(this.files[0]);
                } else {
                    preview.classList.add('hidden');
                }
            });

            document.getElementById('claim-now-btn').addEventListener('click', async () => {
                const formData = new FormData();
                formData.append('action', 'claim_token');
                const response = await fetch('index.php', { method: 'POST', body: formData });
                const result = await response.json();
                showToast(result.message, !result.success);
                if (result.success) {
                    document.getElementById('token-balance').textContent = result.tokens;
                }
            });

            document.body.addEventListener('submit', async function(e) {
                if (e.target.classList.contains('comment-form')) {
                    e.preventDefault();
                    const postId = e.target.dataset.postId;
                    const content = e.target.querySelector('textarea').value;
                    const formData = new FormData();
                    formData.append('action', 'add_comment');
                    formData.append('post_id', postId);
                    formData.append('content', content);
                    const response = await fetch('index.php', { method: 'POST', body: formData });
                    const result = await response.json();
                    showToast(result.message, !result.success);
                    if (result.success) {
                        document.getElementById('token-balance').textContent = result.new_tokens;
                        fetchComments(postId);
                    }
                }
            });
            
            document.body.addEventListener('input', e => {
                if (e.target.classList.contains('comment-box')) {
                    e.target.style.height = 'auto';
                    e.target.style.height = (e.target.scrollHeight) + 'px';
                }
            });
            
            notificationBell.addEventListener('click', async (e) => {
                e.stopPropagation();
                notificationDropdown.classList.toggle('hidden');
                const unreadCount = parseInt(notificationCount.textContent || '0');
                if (!notificationDropdown.classList.contains('hidden') && unreadCount > 0) {
                    notificationList.innerHTML = '<p class="p-4 text-sm text-gray-500">Loading...</p>';
                    const response = await fetch('index.php?action=get_notifications');
                    const result = await response.json();
                    if (result.success && result.notifications.length > 0) {
                        notificationList.innerHTML = '';
                        result.notifications.forEach(notif => {
                            const notifEl = document.createElement('a');
                            notifEl.href = `notifications.php#notif-${notif.id}`;
                            notifEl.className = 'block p-4 hover:bg-gray-50 border-b';
                            notifEl.innerHTML = `
                                <p class="font-semibold text-sm text-gray-800 truncate">${escapeHTML(notif.title)}</p>
                                <p class="text-xs text-gray-500 mt-1">${timeAgo(notif.created_at)}</p>
                            `;
                            notificationList.appendChild(notifEl);
                        });
                    } else {
                        notificationList.innerHTML = '<p class="p-4 text-sm text-gray-500">No new notifications.</p>';
                    }
                    await fetch('index.php', { method: 'POST', headers: {'Content-Type': 'application/x-www-form-urlencoded'}, body: 'action=mark_notifications_read' });
                    notificationCount.textContent = '0';
                    notificationCount.style.display = 'none';
                } else if (!notificationDropdown.classList.contains('hidden')) {
                     notificationList.innerHTML = '<p class="p-4 text-sm text-gray-500">No new notifications.</p>';
                }
            });

            document.addEventListener('click', function(event) {
                if (!notificationBellContainer.contains(event.target)) {
                    notificationDropdown.classList.add('hidden');
                }
            });

            updateUI(currentUser);
            fetchPosts();
        });
    </script>
    <?php if (!empty($settings['ad_code_footer'])): ?>
    <?php echo $settings['ad_code_footer']; ?>
    <?php endif; ?>
</body>
</html>

