<?php
require_once 'db_connection.php';
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

$post_id = (int)($_GET['id'] ?? 0);
$user_id = $_SESSION['user_id'];
$is_admin = isset($_SESSION['is_admin']) && $_SESSION['is_admin'];

// Fetch the post
$stmt = $conn->prepare("SELECT * FROM posts WHERE id = ?");
$stmt->bind_param("i", $post_id);
$stmt->execute();
$post = $stmt->get_result()->fetch_assoc();
$stmt->close();

// Permission check: must be the owner or an admin
if (!$post || (!$is_admin && $post['user_id'] != $user_id)) {
    // Redirect or show an error
    header("Location: " . ($is_admin ? "admin/manage_posts.php" : "myposts.php"));
    exit();
}

$message = '';
$error = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $content = $_POST['content'] ?? '';
    $current_image_url = $post['image_url'];
    
    // Handle image upload
    if (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
        // Delete old image if it exists
        if ($current_image_url && file_exists($current_image_url)) {
            unlink($current_image_url);
        }
        $target_dir = "uploads/";
        $image_name = time() . '_' . basename($_FILES["image"]["name"]);
        $target_file = $target_dir . $image_name;
        if (move_uploaded_file($_FILES["image"]["tmp_name"], $target_file)) {
            $current_image_url = $target_file;
        } else {
            $error = "Failed to upload new image.";
        }
    }

    if (empty($error)) {
        $stmt_update = $conn->prepare("UPDATE posts SET content = ?, image_url = ? WHERE id = ?");
        $stmt_update->bind_param("ssi", $content, $current_image_url, $post_id);
        if ($stmt_update->execute()) {
            $message = "Post updated successfully!";
            // Refresh post data
            $post['content'] = $content;
            $post['image_url'] = $current_image_url;
        } else {
            $error = "Failed to update post.";
        }
        $stmt_update->close();
    }
}
$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Post</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style> body { font-family: 'Inter', sans-serif; background-color: #f0f2f5; } </style>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto p-4 md:p-8 max-w-2xl">
        <div class="bg-white rounded-2xl shadow-xl p-8">
            <div class="flex justify-between items-center mb-6 border-b pb-4">
                 <h1 class="text-3xl font-bold text-gray-800">Edit Post</h1>
                 <a href="<?php echo $is_admin ? 'admin/manage_posts.php' : 'myposts.php'; ?>" class="text-indigo-600 hover:text-indigo-800 font-semibold">← Back</a>
            </div>
            
            <?php if ($message): ?><div class="bg-green-100 text-green-700 p-4 mb-6 rounded-md" role="alert"><p><?php echo $message; ?></p></div><?php endif; ?>
            <?php if ($error): ?><div class="bg-red-100 text-red-700 p-4 mb-6 rounded-md" role="alert"><p><?php echo $error; ?></p></div><?php endif; ?>

            <form method="POST" enctype="multipart/form-data" class="space-y-6">
                <div>
                    <label for="content" class="block text-sm font-medium text-gray-700">Post Content</label>
                    <textarea name="content" id="content" class="mt-1 block w-full p-3 border rounded-lg" rows="6" required><?php echo htmlspecialchars($post['content']); ?></textarea>
                </div>
                <div>
                    <label for="image" class="block text-sm font-medium text-gray-700">Change Picture (optional)</label>
                    <?php if ($post['image_url']): ?>
                        <img src="<?php echo htmlspecialchars($post['image_url']); ?>" class="my-2 rounded-lg max-h-48">
                    <?php endif; ?>
                    <input type="file" name="image" id="image" accept="image/*" class="mt-1 block w-full text-sm file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:font-semibold file:bg-indigo-50 file:text-indigo-700 hover:file:bg-indigo-100"/>
                </div>
                <div class="flex justify-end">
                    <button type="submit" class="w-full bg-indigo-600 text-white font-semibold py-2.5 px-6 rounded-lg hover:bg-indigo-700">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
</body>
</html>
