<?php
require_once 'db_connection.php';
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}
$user_id = $_SESSION['user_id'];

$message = '';
$error = '';

// Handle Purchase Submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_purchase'])) {
    $package_id = (int)$_POST['package_id'];
    $transaction_id = trim($_POST['transaction_id']);

    if (empty($transaction_id) || $package_id <= 0) {
        $error = "Please provide a valid Transaction ID.";
    } else {
        // Check for existing pending request
        $stmt_check = $conn->prepare("SELECT id FROM package_purchases WHERE user_id = ? AND status = 'pending'");
        $stmt_check->bind_param("i", $user_id);
        $stmt_check->execute();
        if ($stmt_check->get_result()->num_rows > 0) {
            $error = "You already have a pending package request. Please wait for it to be processed.";
        } else {
            $stmt = $conn->prepare("INSERT INTO package_purchases (user_id, package_id, transaction_id) VALUES (?, ?, ?)");
            $stmt->bind_param("iis", $user_id, $package_id, $transaction_id);
            if ($stmt->execute()) {
                $message = "Your purchase request has been submitted successfully! Please wait for admin approval.";
            } else {
                $error = "Failed to submit your request.";
            }
            $stmt->close();
        }
        $stmt_check->close();
    }
}

// Fetch user's current package and packages list
$packages = $conn->query("SELECT * FROM packages ORDER BY price ASC")->fetch_all(MYSQLI_ASSOC);
$user_package_sql = "SELECT p.name FROM users u JOIN packages p ON u.package_id = p.id WHERE u.id = ?";
$stmt_user_pkg = $conn->prepare($user_package_sql);
$stmt_user_pkg->bind_param("i", $user_id);
$stmt_user_pkg->execute();
$user_package_name = $stmt_user_pkg->get_result()->fetch_assoc()['name'] ?? 'Free';
$stmt_user_pkg->close();

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Buy a Package</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style> body { font-family: 'Inter', sans-serif; background-color: #f0f2f5; } </style>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto p-4 md:p-8 max-w-5xl">
        <div class="bg-white rounded-2xl shadow-xl p-8">
            <div class="flex justify-between items-center mb-6 border-b pb-4">
                 <div>
                    <h1 class="text-3xl font-bold text-gray-800">Upgrade Your Plan</h1>
                    <p class="text-gray-500 mt-1">Your Current Plan: <span class="font-bold text-indigo-600"><?php echo htmlspecialchars($user_package_name); ?></span></p>
                 </div>
                 <a href="index.php" class="text-indigo-600 hover:text-indigo-800 font-semibold">← Back to Dashboard</a>
            </div>

            <?php if ($message): ?>
            <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-6 rounded-md" role="alert"><p><?php echo $message; ?></p></div>
            <?php endif; ?>
            <?php if ($error): ?>
            <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-6 rounded-md" role="alert"><p><?php echo $error; ?></p></div>
            <?php endif; ?>

            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                <?php foreach ($packages as $pkg): ?>
                <div class="border-2 <?php echo $pkg['name'] === 'Gold' ? 'border-indigo-500' : 'border-gray-200'; ?> rounded-xl p-6 flex flex-col">
                    <h3 class="text-lg font-semibold text-indigo-700"><?php echo htmlspecialchars($pkg['name']); ?></h3>
                    <p class="text-4xl font-extrabold my-4">৳<?php echo number_format($pkg['price']); ?></p>
                    <ul class="space-y-2 text-gray-600 mb-6 flex-grow">
                        <li class="flex items-center"><i class="fas fa-check-circle text-green-500 mr-2"></i><?php echo htmlspecialchars($pkg['daily_post_limit']); ?> Posts/Day</li>
                        <li class="flex items-center"><i class="fas fa-check-circle text-green-500 mr-2"></i><?php echo htmlspecialchars($pkg['daily_comment_limit']); ?> Comments/Day</li>
                        <li class="flex items-center"><i class="fas fa-check-circle text-green-500 mr-2"></i><?php echo htmlspecialchars($pkg['post_token_reward']); ?> Tokens/Post</li>
                        <li class="flex items-center"><i class="fas fa-check-circle text-green-500 mr-2"></i><?php echo htmlspecialchars($pkg['comment_token_reward']); ?> Tokens/Comment</li>
                    </ul>
                    <button onclick="openPurchaseModal(<?php echo htmlspecialchars(json_encode($pkg)); ?>)" class="w-full mt-auto <?php echo $pkg['name'] === 'Gold' ? 'bg-indigo-600 text-white' : 'bg-indigo-100 text-indigo-700'; ?> font-semibold py-3 px-6 rounded-lg hover:opacity-90">
                        Choose Plan
                    </button>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
    
    <!-- Purchase Modal -->
    <div id="purchase-modal" class="hidden fixed inset-0 bg-gray-900 bg-opacity-75 flex items-center justify-center z-50 p-4">
        <div class="bg-white rounded-lg shadow-xl w-full max-w-md relative p-8">
            <button onclick="closePurchaseModal()" class="absolute top-4 right-4 text-gray-500 hover:text-gray-800"><i class="fas fa-times text-2xl"></i></button>
            <h2 class="text-2xl font-bold mb-4">Purchase <span id="modal-pkg-name" class="text-indigo-600"></span></h2>
            <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 rounded-md mb-6">
                <p class="font-semibold">Payment Instructions:</p>
                <p class="text-sm">Please send <strong id="modal-pkg-price"></strong> to the following address and enter the Transaction ID below.</p>
                <p id="modal-pkg-address" class="mt-2 font-mono bg-gray-100 p-2 rounded"></p>
            </div>
            <form method="POST">
                <input type="hidden" name="package_id" id="modal-pkg-id">
                <div>
                    <label for="transaction_id" class="block text-sm font-medium text-gray-700">Transaction ID (TrxID)</label>
                    <input type="text" name="transaction_id" id="transaction_id" required class="mt-1 block w-full rounded-md border-gray-300 shadow-sm" placeholder="Enter your payment TrxID here">
                </div>
                <button type="submit" name="submit_purchase" class="w-full mt-6 bg-indigo-600 text-white font-semibold py-2.5 px-4 rounded-lg hover:bg-indigo-700">Submit for Approval</button>
            </form>
        </div>
    </div>

    <script>
        const purchaseModal = document.getElementById('purchase-modal');
        function openPurchaseModal(pkg) {
            document.getElementById('modal-pkg-name').textContent = pkg.name;
            document.getElementById('modal-pkg-price').textContent = `৳${pkg.price}`;
            document.getElementById('modal-pkg-address').textContent = pkg.payment_address;
            document.getElementById('modal-pkg-id').value = pkg.id;
            purchaseModal.classList.remove('hidden');
        }
        function closePurchaseModal() {
            purchaseModal.classList.add('hidden');
        }
    </script>
</body>
</html>
