<?php
require_once '../db_connection.php';
if (!isset($_SESSION['user_id']) || !isset($_SESSION['is_admin']) || !$_SESSION['is_admin']) {
    header("Location: index.php");
    exit();
}

$message = '';
$error = '';
$search_email = trim($_GET['search_email'] ?? '');

// Handle Block/Unblock actions
if (isset($_GET['action']) && isset($_GET['user_id'])) {
    $user_id_to_update = (int)$_GET['user_id'];
    $action = $_GET['action'];
    $new_status = '';

    if ($action === 'block') {
        $new_status = 'blocked';
    } elseif ($action === 'unblock') {
        $new_status = 'active';
    }

    if (!empty($new_status) && $user_id_to_update !== $_SESSION['user_id']) { // Admins cannot block themselves
        $stmt = $conn->prepare("UPDATE users SET status = ? WHERE id = ?");
        $stmt->bind_param("si", $new_status, $user_id_to_update);
        if ($stmt->execute()) {
            $message = "User status updated successfully!";
        } else {
            $error = "Failed to update user status.";
        }
        $stmt->close();
    } elseif ($user_id_to_update === $_SESSION['user_id']) {
        $error = "You cannot block your own account.";
    }
}

// Fetch all users with search functionality
$sql = "SELECT id, name, email, tokens, status, ip_address, created_at FROM users";
$params = [];
$types = '';

if (!empty($search_email)) {
    $sql .= " WHERE email LIKE ?";
    $search_param = "%" . $search_email . "%";
    $params[] = &$search_param;
    $types .= 's';
}

$sql .= " ORDER BY created_at DESC";
$stmt_users = $conn->prepare($sql);

if (!empty($params)) {
    $stmt_users->bind_param($types, ...$params);
}

$stmt_users->execute();
$result = $stmt_users->get_result();

$all_users = [];
while ($row = $result->fetch_assoc()) {
    $all_users[] = $row;
}
$stmt_users->close();
$conn->close();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Users</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Inter', sans-serif; background-color: #f0f2f5; }
        .tab-button.active { border-bottom-color: #4f46e5; color: #4f46e5; font-weight: 600; }
    </style>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto p-4 md:p-8 max-w-7xl">
         <div class="bg-white rounded-2xl shadow-xl p-8">
            <div class="flex justify-between items-center mb-6 border-b pb-4">
                 <h1 class="text-3xl font-bold text-gray-800">Manage Users</h1>
                 <a href="admindashboard.php" class="text-indigo-600 hover:text-indigo-800 font-semibold">← Back to Dashboard</a>
            </div>

            <?php if ($message): ?>
                <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-6 rounded-md" role="alert"><p><?php echo $message; ?></p></div>
            <?php endif; ?>
             <?php if ($error): ?>
                <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-6 rounded-md" role="alert"><p><?php echo $error; ?></p></div>
            <?php endif; ?>

            <!-- Search Form -->
            <div class="mb-6">
                <form method="GET" class="flex items-center space-x-2">
                    <input type="email" name="search_email" value="<?php echo htmlspecialchars($search_email); ?>" placeholder="Search by user email..." class="w-full md:w-1/3 px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500">
                    <button type="submit" class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-indigo-600 hover:bg-indigo-700">Search</button>
                    <?php if(!empty($search_email)): ?>
                        <a href="manage_users.php" class="text-gray-500 hover:text-gray-700">Clear</a>
                    <?php endif; ?>
                </form>
            </div>


            <!-- Tabs Navigation -->
            <div class="border-b border-gray-200 mb-6">
                <nav class="flex space-x-8" aria-label="Tabs">
                    <button class="tab-button active py-4 px-1 text-sm font-medium" data-target="active">Active Users</button>
                    <button class="tab-button py-4 px-1 text-sm font-medium text-gray-500 hover:text-gray-700" data-target="blocked">Blocked Users</button>
                </nav>
            </div>

            <!-- Tabs Content -->
            <div id="tab-content">
                <?php foreach (['active', 'blocked'] as $status): ?>
                <div id="<?php echo $status; ?>" class="tab-pane hidden">
                     <div class="overflow-x-auto rounded-lg border">
                         <table class="min-w-full divide-y divide-gray-200">
                             <thead class="bg-gray-50">
                                 <tr>
                                     <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">User</th>
                                     <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Tokens</th>
                                     <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">IP Address</th>
                                     <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Joined</th>
                                     <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase">Actions</th>
                                 </tr>
                             </thead>
                             <tbody class="bg-white divide-y divide-gray-200">
                                 <?php 
                                    $users_for_status = array_filter($all_users, fn($u) => $u['status'] === $status);
                                    if (empty($users_for_status)): 
                                 ?>
                                     <tr><td colspan="5" class="px-6 py-10 text-center text-gray-500">No <?php echo $status; ?> users found.</td></tr>
                                 <?php else: ?>
                                     <?php foreach ($users_for_status as $user): ?>
                                         <tr>
                                             <td class="px-6 py-4 whitespace-nowrap">
                                                <div class="font-medium text-gray-900"><?php echo htmlspecialchars($user['name']); ?></div>
                                                <div class="text-sm text-gray-500"><?php echo htmlspecialchars($user['email']); ?></div>
                                             </td>
                                             <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-800"><?php echo number_format($user['tokens']); ?></td>
                                             <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo htmlspecialchars($user['ip_address']); ?></td>
                                             <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo date("d M, Y", strtotime($user['created_at'])); ?></td>
                                             <td class="px-6 py-4 whitespace-nowrap text-sm text-center space-x-2">
                                                 <a href="edit_user.php?id=<?php echo $user['id']; ?>" class="text-indigo-600 hover:text-indigo-900">Edit</a>
                                                 <?php if ($status === 'active'): ?>
                                                     <a href="manage_users.php?action=block&user_id=<?php echo $user['id']; ?>" onclick="return confirm('Are you sure you want to block this user?');" class="text-red-600 hover:text-red-900">Block</a>
                                                 <?php else: ?>
                                                      <a href="manage_users.php?action=unblock&user_id=<?php echo $user['id']; ?>" class="text-green-600 hover:text-green-900">Unblock</a>
                                                 <?php endif; ?>
                                             </td>
                                         </tr>
                                     <?php endforeach; ?>
                                 <?php endif; ?>
                             </tbody>
                         </table>
                     </div>
                </div>
                <?php endforeach; ?>
            </div>
         </div>
    </div>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const tabs = document.querySelectorAll('.tab-button');
            const panes = document.querySelectorAll('.tab-pane');
            document.getElementById('active').classList.remove('hidden');
            tabs.forEach(tab => {
                tab.addEventListener('click', function() {
                    tabs.forEach(item => item.classList.remove('active'));
                    panes.forEach(pane => pane.classList.add('hidden'));
                    this.classList.add('active');
                    document.getElementById(this.dataset.target).classList.remove('hidden');
                });
            });
        });
    </script>
</body>
</html>

