<?php
require_once '../db_connection.php';
if (!isset($_SESSION['user_id']) || !isset($_SESSION['is_admin']) || !$_SESSION['is_admin']) {
    header("Location: index.php");
    exit();
}

$message = '';
$error = '';

// Handle Add/Edit Package
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_package'])) {
    $package_id = (int)($_POST['package_id'] ?? 0);
    $name = trim($_POST['name']);
    $price = (float)$_POST['price'];
    $post_limit = (int)$_POST['post_limit'];
    $comment_limit = (int)$_POST['comment_limit'];
    $post_reward = (int)$_POST['post_reward'];
    $comment_reward = (int)$_POST['comment_reward'];
    $payment_address = trim($_POST['payment_address']);

    if ($package_id > 0) { // Update
        $stmt = $conn->prepare("UPDATE packages SET name=?, price=?, daily_post_limit=?, daily_comment_limit=?, post_token_reward=?, comment_token_reward=?, payment_address=? WHERE id=?");
        $stmt->bind_param("sdiiiisi", $name, $price, $post_limit, $comment_limit, $post_reward, $comment_reward, $payment_address, $package_id);
    } else { // Insert
        $stmt = $conn->prepare("INSERT INTO packages (name, price, daily_post_limit, daily_comment_limit, post_token_reward, comment_token_reward, payment_address) VALUES (?, ?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("sdiiiis", $name, $price, $post_limit, $comment_limit, $post_reward, $comment_reward, $payment_address);
    }
    if ($stmt->execute()) {
        $message = "Package saved successfully!";
    } else {
        $error = "Error saving package.";
    }
    $stmt->close();
}

// Handle Delete Package
if (isset($_GET['delete_package'])) {
    $package_id = (int)$_GET['delete_package'];
    $stmt = $conn->prepare("DELETE FROM packages WHERE id = ?");
    $stmt->bind_param("i", $package_id);
    if($stmt->execute()){
        $message = "Package deleted successfully.";
    } else {
        $error = "Could not delete package. It might be in use by users.";
    }
    $stmt->close();
}

// Handle Purchase Approval/Rejection
if (isset($_GET['action']) && isset($_GET['purchase_id'])) {
    $purchase_id = (int)$_GET['purchase_id'];
    $action = $_GET['action'];
    $new_status = ($action === 'approve') ? 'approved' : 'rejected';

    $stmt_update = $conn->prepare("UPDATE package_purchases SET status = ? WHERE id = ? AND status = 'pending'");
    $stmt_update->bind_param("si", $new_status, $purchase_id);
    
    if ($stmt_update->execute() && $stmt_update->affected_rows > 0) {
        if ($new_status === 'approved') {
            $stmt_get = $conn->prepare("SELECT user_id, package_id FROM package_purchases WHERE id = ?");
            $stmt_get->bind_param("i", $purchase_id);
            $stmt_get->execute();
            $purchase = $stmt_get->get_result()->fetch_assoc();
            $stmt_get->close();

            $stmt_user = $conn->prepare("UPDATE users SET package_id = ? WHERE id = ?");
            $stmt_user->bind_param("ii", $purchase['package_id'], $purchase['user_id']);
            $stmt_user->execute();
            $stmt_user->close();
        }
        $message = "Purchase request has been {$new_status}.";
    } else {
        $error = "Failed to update purchase request. It may have been processed already.";
    }
    $stmt_update->close();
}

// Fetch all packages and purchases
$packages = $conn->query("SELECT * FROM packages ORDER BY price ASC")->fetch_all(MYSQLI_ASSOC);
$purchases_sql = "SELECT pp.id, pp.transaction_id, pp.status, pp.created_at, u.name as user_name, p.name as package_name 
                  FROM package_purchases pp 
                  JOIN users u ON pp.user_id = u.id 
                  JOIN packages p ON pp.package_id = p.id 
                  ORDER BY pp.created_at DESC";
$all_purchases = $conn->query($purchases_sql)->fetch_all(MYSQLI_ASSOC);

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Packages</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Inter', sans-serif; background-color: #f0f2f5; }
        .tab-button.active { border-bottom-color: #4f46e5; color: #4f46e5; font-weight: 600; }
    </style>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto p-4 md:p-8 max-w-7xl">
        <div class="bg-white rounded-2xl shadow-xl p-8 space-y-12">
            <div class="flex justify-between items-center border-b pb-4">
                 <h1 class="text-3xl font-bold text-gray-800">Manage Packages & Purchases</h1>
                 <a href="admindashboard.php" class="text-indigo-600 hover:text-indigo-800 font-semibold">← Back to Dashboard</a>
            </div>

            <?php if ($message): ?>
            <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 rounded-md" role="alert"><p><?php echo $message; ?></p></div>
            <?php endif; ?>
            <?php if ($error): ?>
            <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-6 rounded-md" role="alert"><p><?php echo $error; ?></p></div>
            <?php endif; ?>

            <!-- Section: Manage Packages -->
            <div>
                <h2 class="text-2xl font-semibold text-gray-700 mb-4">Packages</h2>
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
                    <?php foreach ($packages as $pkg): ?>
                    <div class="bg-gray-50 p-4 rounded-lg border">
                        <div class="flex justify-between items-start">
                             <h3 class="text-lg font-bold text-indigo-700"><?php echo htmlspecialchars($pkg['name']); ?></h3>
                             <div>
                                <a href="?edit_package=<?php echo $pkg['id']; ?>" class="text-blue-500 hover:text-blue-700 mr-2"><i class="fas fa-edit"></i></a>
                                <a href="?delete_package=<?php echo $pkg['id']; ?>" onclick="return confirm('Are you sure?');" class="text-red-500 hover:text-red-700"><i class="fas fa-trash"></i></a>
                             </div>
                        </div>
                        <p class="text-2xl font-bold">৳<?php echo htmlspecialchars($pkg['price']); ?></p>
                        <ul class="text-sm text-gray-600 mt-2 space-y-1">
                            <li>Post Limit: <?php echo htmlspecialchars($pkg['daily_post_limit']); ?>/day</li>
                            <li>Comment Limit: <?php echo htmlspecialchars($pkg['daily_comment_limit']); ?>/day</li>
                            <li>Post Reward: <?php echo htmlspecialchars($pkg['post_token_reward']); ?> tokens</li>
                            <li>Comment Reward: <?php echo htmlspecialchars($pkg['comment_token_reward']); ?> tokens</li>
                        </ul>
                    </div>
                    <?php endforeach; ?>
                </div>

                <!-- Add/Edit Form -->
                <?php
                $edit_pkg = null;
                if(isset($_GET['edit_package'])){
                    $edit_id = (int)$_GET['edit_package'];
                    foreach($packages as $p){ if($p['id'] == $edit_id) $edit_pkg = $p; }
                }
                ?>
                <form method="POST" class="bg-gray-50 p-6 rounded-lg border">
                    <h3 class="text-xl font-semibold mb-4"><?php echo $edit_pkg ? 'Edit Package' : 'Add New Package'; ?></h3>
                    <input type="hidden" name="package_id" value="<?php echo $edit_pkg['id'] ?? 0; ?>">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <input name="name" placeholder="Package Name" value="<?php echo htmlspecialchars($edit_pkg['name'] ?? ''); ?>" class="p-2 border rounded" required>
                        <input name="price" type="number" step="0.01" placeholder="Price (BDT)" value="<?php echo htmlspecialchars($edit_pkg['price'] ?? ''); ?>" class="p-2 border rounded" required>
                        <input name="post_limit" type="number" placeholder="Daily Post Limit" value="<?php echo htmlspecialchars($edit_pkg['daily_post_limit'] ?? ''); ?>" class="p-2 border rounded" required>
                        <input name="comment_limit" type="number" placeholder="Daily Comment Limit" value="<?php echo htmlspecialchars($edit_pkg['daily_comment_limit'] ?? ''); ?>" class="p-2 border rounded" required>
                        <input name="post_reward" type="number" placeholder="Post Token Reward" value="<?php echo htmlspecialchars($edit_pkg['post_token_reward'] ?? ''); ?>" class="p-2 border rounded" required>
                        <input name="comment_reward" type="number" placeholder="Comment Token Reward" value="<?php echo htmlspecialchars($edit_pkg['comment_token_reward'] ?? ''); ?>" class="p-2 border rounded" required>
                        <textarea name="payment_address" placeholder="Payment Address/Instructions" class="p-2 border rounded md:col-span-2" required><?php echo htmlspecialchars($edit_pkg['payment_address'] ?? ''); ?></textarea>
                    </div>
                    <button type="submit" name="save_package" class="mt-4 bg-indigo-600 text-white font-semibold py-2 px-6 rounded-lg hover:bg-indigo-700">Save Package</button>
                    <?php if ($edit_pkg): ?><a href="manage_packages.php" class="mt-4 ml-2 text-gray-600">Cancel Edit</a><?php endif; ?>
                </form>
            </div>

            <!-- Section: Manage Purchases -->
            <div>
                <h2 class="text-2xl font-semibold text-gray-700 mb-4">Purchase Requests</h2>
                <div class="border-b border-gray-200 mb-6">
                    <nav class="flex space-x-8" aria-label="Tabs">
                        <button class="tab-button active py-4 px-1" data-target="pending">Pending</button>
                        <button class="tab-button py-4 px-1 text-gray-500" data-target="approved">Approved</button>
                        <button class="tab-button py-4 px-1 text-gray-500" data-target="rejected">Rejected</button>
                    </nav>
                </div>
                <div id="tab-content">
                    <?php foreach (['pending', 'approved', 'rejected'] as $status): ?>
                    <div id="<?php echo $status; ?>" class="tab-pane hidden">
                        <div class="overflow-x-auto rounded-lg border">
                        <table class="min-w-full divide-y divide-gray-200">
                             <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">User & Package</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Transaction ID</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                                    <?php if ($status === 'pending'): ?>
                                    <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase">Actions</th>
                                    <?php endif; ?>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                            <?php 
                            $purchases_for_status = array_filter($all_purchases, fn($p) => $p['status'] === $status);
                            if (empty($purchases_for_status)):
                            ?>
                                <tr><td colspan="4" class="px-6 py-10 text-center text-gray-500">No <?php echo $status; ?> requests.</td></tr>
                            <?php else: foreach ($purchases_for_status as $p): ?>
                                <tr>
                                    <td class="px-6 py-4">
                                        <div class="font-medium text-gray-900"><?php echo htmlspecialchars($p['user_name']); ?></div>
                                        <div class="text-sm text-indigo-600"><?php echo htmlspecialchars($p['package_name']); ?></div>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-600"><?php echo htmlspecialchars($p['transaction_id']); ?></td>
                                    <td class="px-6 py-4 text-sm text-gray-500"><?php echo date("d M, Y", strtotime($p['created_at'])); ?></td>
                                    <?php if ($status === 'pending'): ?>
                                    <td class="px-6 py-4 text-center space-x-2">
                                        <a href="?action=approve&purchase_id=<?php echo $p['id']; ?>" class="text-white bg-green-600 hover:bg-green-700 px-3 py-1 rounded-full text-xs">Approve</a>
                                        <a href="?action=reject&purchase_id=<?php echo $p['id']; ?>" onclick="return confirm('Are you sure?');" class="text-white bg-red-600 hover:bg-red-700 px-3 py-1 rounded-full text-xs">Reject</a>
                                    </td>
                                    <?php endif; ?>
                                </tr>
                            <?php endforeach; endif; ?>
                            </tbody>
                        </table>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const tabs = document.querySelectorAll('.tab-button');
        const panes = document.querySelectorAll('.tab-pane');
        document.getElementById('pending').classList.remove('hidden');
        tabs.forEach(tab => {
            tab.addEventListener('click', function() {
                tabs.forEach(item => item.classList.remove('active', 'text-indigo-600'));
                panes.forEach(pane => pane.classList.add('hidden'));
                this.classList.add('active', 'text-indigo-600');
                document.getElementById(this.dataset.target).classList.remove('hidden');
            });
        });
    });
    </script>
</body>
</html>
